<?php

namespace App\Http\Controllers;

use App\Models\Unit;
use Illuminate\Http\Request;

class UnitController extends Controller
{
    /**
     * Display all units.
     */
    public function index(Request $request)
    {
        try {
            $query = Unit::query();

            // 🔍 Search by name or short_name
            if ($request->filled('search')) {
                $search = $request->search;
                $query->where(function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                      ->orWhere('short_name', 'like', "%{$search}%");
                });
            }

            // 🧩 Status filter
            if ($request->filled('status') && in_array($request->status, ['active', 'inactive'])) {
                $query->where('status', $request->status);
            }

            // ↕️ Sorting (default: desc)
            $order = $request->get('order', 'desc');
            $query->orderBy('id', $order);

            // 📄 Pagination (default: 10 per page)
            $perPage = $request->get('per_page', 10);
            $units = $query->paginate($perPage);

            // Transform unit data
            $units->getCollection()->transform(function ($unit) {
                return [
                    'id' => $unit->id,
                    'name' => $unit->name,
                    'short_name' => $unit->short_name,
                    'description' => $unit->description,
                    'status' => $unit->status,
                    'created_at' => $unit->created_at->toDateTimeString(),
                    'updated_at' => $unit->updated_at->toDateTimeString(),
                ];
            });

            return response()->json([
                'status' => true,
                'message' => 'Units retrieved successfully',
                'pagination' => [
                    'limit_page' => $perPage,
                    'total_count' => $units->total(),
                    'total_page' => $units->lastPage(),
                    'current_page' => $units->currentPage(),
                    'current_page_count' => $units->count(),
                    'next_page' => $units->hasMorePages() ? $units->currentPage() + 1 : null,
                    'previous_page' => $units->onFirstPage() ? null : $units->currentPage() - 1,
                ],
                'data' => $units->items(),
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Store a newly created unit.
     */
    public function store(Request $request)
    {
        try {
            $request->validate([
                'name' => 'required|string|max:255|unique:units,name',
                'short_name' => 'required|string|max:10|unique:units,short_name',
                'description' => 'nullable|string',
                'status' => 'in:active,inactive'
            ]);

            $unit = Unit::create([
                'name' => $request->name,
                'short_name' => $request->short_name,
                'description' => $request->description,
                'status' => $request->status ?? 'active'
            ]);

            return response()->json([
                'status' => true,
                'message' => 'Unit created successfully',
                'data' => $unit
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Display the specified unit.
     */
    public function show($id)
    {
        try {
            $unit = Unit::find($id);

            if (!$unit) {
                return response()->json(['status' => false, 'message' => 'Unit not found'], 404);
            }

            return response()->json(['status' => true, 'data' => $unit]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Update the specified unit.
     */
    public function update(Request $request, $id)
    {
        try {
            $unit = Unit::find($id);

            if (!$unit) {
                return response()->json(['status' => false, 'message' => 'Unit not found'], 404);
            }

            $request->validate([
                'name' => 'sometimes|string|max:255|unique:units,name,' . $id,
                'short_name' => 'sometimes|string|max:10|unique:units,short_name,' . $id,
                'description' => 'nullable|string',
                'status' => 'in:active,inactive'
            ]);

            $unit->update([
                'name' => $request->name ?? $unit->name,
                'short_name' => $request->short_name ?? $unit->short_name,
                'description' => $request->description ?? $unit->description,
                'status' => $request->status ?? $unit->status
            ]);

            return response()->json([
                'status' => true,
                'message' => 'Unit updated successfully',
                'data' => $unit
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Remove the specified unit.
     */
    public function destroy($id)
    {
        try {
            $unit = Unit::find($id);

            if (!$unit) {
                return response()->json(['status' => false, 'message' => 'Unit not found'], 404);
            }

            $unit->delete();

            return response()->json([
                'status' => true,
                'message' => 'Unit deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }
}
