<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\Sales;
use Illuminate\Http\Request;

class SalesController extends Controller
{
    /**
     * Display a listing of the sales.
     */
    public function index(Request $request)
    {
        $query = Sales::query();

        // Search by customer_name, customer_phone, or sales_id
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('customer_name', 'like', "%{$search}%")
                  ->orWhere('customer_phone', 'like', "%{$search}%")
                  ->orWhere('sales_id', 'like', "%{$search}%");
            });
        }

        // Pagination
        $perPage = $request->get('per_page', 10);
        $sales = $query->orderBy('id', 'desc')->paginate($perPage);

        // Decode products json for each result
        $data = [];

        // Preload category/brand/etc for resolving names efficiently
        $categoryIds = [];
        $subcatIds = [];
        $brandIds = [];
        $unitIds = [];
        $colorIds = [];
        $sizeIds = [];

        // 1st pass: collect ids from all product lines
        foreach ($sales->items() as $item) {
            $products = json_decode($item->products, true);
            if (is_array($products)) {
                $productIds = array_column($products, 'product_id');
                $productsFromDb = Product::whereIn('id', $productIds)->get();

                foreach ($productsFromDb as $p) {
                    if ($p->cat_id) $categoryIds[] = $p->cat_id;
                    if ($p->subcat_id) $subcatIds[] = $p->subcat_id;
                    if ($p->brand_id) $brandIds[] = $p->brand_id;
                    if ($p->unit_id) $unitIds[] = $p->unit_id;
                    if ($p->color_id) $colorIds[] = $p->color_id;
                    if ($p->size_ids) {
                        foreach (explode(',', $p->size_ids) as $sid) {
                            $sizeIds[] = trim($sid);
                        }
                    }
                }
            }
        }

        // Load names for all collected IDs in batch
        $categoryNames = \App\Models\Category::whereIn('id', array_unique($categoryIds))->pluck('name', 'id');
        $subcatNames = \App\Models\Subcategory::whereIn('id', array_unique($subcatIds))->pluck('name', 'id');
        $brandNames = \App\Models\Brand::whereIn('id', array_unique($brandIds))->pluck('name', 'id');
        $unitNames = \App\Models\Unit::whereIn('id', array_unique($unitIds))->pluck('name', 'id');
        $colorNames = \App\Models\Color::whereIn('id', array_unique($colorIds))->pluck('name', 'id');
        $sizeNames = \App\Models\Size::whereIn('id', array_unique($sizeIds))->pluck('name', 'id');

        foreach ($sales->items() as $item) {
            $itemArr = $item->toArray();
            $products = json_decode($itemArr['products'], true);

            $detailedProducts = [];
            if (is_array($products)) {
                $productIds = array_column($products, 'product_id');
                $productsFromDb = Product::whereIn('id', $productIds)->get()->keyBy('id');

                foreach ($products as $prodItem) {
                    $prodDetail = $prodItem;
                    if (isset($prodItem['product_id']) && isset($productsFromDb[$prodItem['product_id']])) {
                        $product = $productsFromDb[$prodItem['product_id']];
                        // Build enriched product details with names
                        $prodDetail['product_details'] = [
                            'id' => $product->id,
                            'name' => $product->name,
                            'sku' => $product->sku,
                            'barcode' => $product->barcode,
                            'image' => $product->image ? asset('storage/' . $product->image) : null,
                            'category' => $product->cat_id ? ($categoryNames[$product->cat_id] ?? null) : null,
                            'subcategory' => $product->subcat_id ? ($subcatNames[$product->subcat_id] ?? null) : null,
                            'brand' => $product->brand_id ? ($brandNames[$product->brand_id] ?? null) : null,
                            'unit' => $product->unit_id ? ($unitNames[$product->unit_id] ?? null) : null,
                            'color' => $product->color_id ? ($colorNames[$product->color_id] ?? null) : null,
                            'price' => $product->price,
                            'offer_price' => $product->offer_price,
                            'stock' => $product->stock,
                            'alert_quantity' => $product->alert_quantity,
                            'weight' => $product->weight,
                            'sizes' => $product->size_ids ? collect(explode(',', $product->size_ids))->map(fn($sid) => $sizeNames[$sid] ?? null)->filter()->values()->all() : [],
                            'material' => $product->material,
                            'origin' => $product->origin,
                            'description' => $product->description,
                            'gallery_images' => $product->gallery_images
                                ? array_map(fn($img) => asset('storage/' . $img), json_decode($product->gallery_images))
                                : [],
                            'extra_status' => $product->extra_status,
                            'status' => $product->status,
                            'created_at' => $product->created_at ? $product->created_at->toDateTimeString() : null,
                            'updated_at' => $product->updated_at ? $product->updated_at->toDateTimeString() : null,
                        ];
                    } else {
                        $prodDetail['product_details'] = null;
                    }
                    $detailedProducts[] = $prodDetail;
                }
            }

            $itemArr['products'] = $detailedProducts;
            $data[] = $itemArr;
        }


        return response()->json([
            'status' => true,
            'message' => 'Sales retrieved successfully',
            'data' => $data,
            'pagination' => [
                'limit_page' => $perPage,
                'total_count' => $sales->total(),
                'total_page' => $sales->lastPage(),
                'current_page' => $sales->currentPage(),
                'current_page_count' => $sales->count(),
                'next_page' => $sales->hasMorePages() ? $sales->currentPage() + 1 : null,
                'previous_page' => $sales->onFirstPage() ? null : $sales->currentPage() - 1,
            ],
        ]);
    }

    /**
     * Store a newly created sale in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            // 'sales_id' is now auto-generated
            'customer_name' => 'required|string|max:255',
            'customer_phone' => 'required|string|max:255',
            'products' => 'required|array',
            'total_amount' => 'required|string',
            'vat' => 'required|string',
            'discount' => 'required|string',
            'grand_total' => 'required|string',
            'paid_by' => 'required|string',
        ]);

        // Generate sales_id: "SLS" + 12 chars of capital letters and numbers
        do {
            $salesId = 'SLS' . strtoupper(substr(bin2hex(random_bytes(12)), 0, 12));
        } while (Sales::where('sales_id', $salesId)->exists());

        $validated['sales_id'] = $salesId;

        // Store products as JSON
        $validated['products'] = json_encode($validated['products']);

        $sale = Sales::create($validated);

        $saleArray = $sale->toArray();
        $saleArray['products'] = json_decode($saleArray['products'], true);

        return response()->json([
            'status' => true,
            'message' => 'Sale created successfully',
            'data' => $saleArray
        ], 201);
    }

    /**
     * Display the specified sale.
     */
    public function show($id)
    {
        $sale = Sales::find($id);

        if (!$sale) {
            return response()->json([
                'status' => false,
                'message' => 'Sale not found'
            ], 404);
        }

        $saleArray = $sale->toArray();
        $saleArray['products'] = json_decode($saleArray['products'], true);

        return response()->json([
            'status' => true,
            'message' => 'Sale retrieved successfully',
            'data' => $saleArray
        ]);
    }

    /**
     * Update the specified sale in storage.
     */
    public function update(Request $request, $id)
    {
        $sale = Sales::find($id);
        if (!$sale) {
            return response()->json([
                'status' => false,
                'message' => 'Sale not found'
            ], 404);
        }

        $validated = $request->validate([
            'sales_id' => 'sometimes|required|string|unique:sales,sales_id,' . $sale->id,
            'customer_name' => 'sometimes|required|string|max:255',
            'customer_phone' => 'sometimes|required|string|max:255',
            'products' => 'sometimes|required|array',
            'total_amount' => 'sometimes|required|string',
            'vat' => 'sometimes|required|string',
            'discount' => 'sometimes|required|string',
            'grand_total' => 'sometimes|required|string',
            'paid_by' => 'sometimes|required|string',
        ]);

        if (isset($validated['products'])) {
            $validated['products'] = json_encode($validated['products']);
        }

        $sale->update($validated);

        $saleArray = $sale->toArray();
        $saleArray['products'] = json_decode($saleArray['products'], true);

        return response()->json([
            'status' => true,
            'message' => 'Sale updated successfully',
            'data' => $saleArray
        ]);
    }

    /**
     * Remove the specified sale from storage.
     */
    public function destroy($id)
    {
        $sale = Sales::find($id);
        if (!$sale) {
            return response()->json([
                'status' => false,
                'message' => 'Sale not found'
            ], 404);
        }

        $sale->delete();

        return response()->json([
            'status' => true,
            'message' => 'Sale deleted successfully'
        ]);
    }

    /**
     * Provide a report of sales with optional date filter and search.
     * Filters:
     *  - from_date: (Y-m-d) sales_date >= from_date
     *  - to_date: (Y-m-d) sales_date <= to_date
     *  - search: (customer_name, customer_phone, or sales_id)
     */
    public function report(Request $request)
    {
        $query = Sales::query();

        // Support 'days' input for quick period filtering (from today back N days)
        if ($request->filled('days')) {
            $days = (int)$request->get('days');
            if ($days > 0) {
                $fromDate = now()->subDays($days)->startOfDay();
                $toDate = now()->endOfDay();
                $query->whereBetween('sales_date', [$fromDate, $toDate]);
            }
        }

        // Filter by date range (overrides 'days' if present)
        if ($request->filled('from_date')) {
            $query->whereDate('sales_date', '>=', $request->get('from_date'));
        }
        if ($request->filled('to_date')) {
            $query->whereDate('sales_date', '<=', $request->get('to_date'));
        }

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('customer_name', 'like', "%{$search}%")
                  ->orWhere('customer_phone', 'like', "%{$search}%")
                  ->orWhere('sales_id', 'like', "%{$search}%");
            });
        }

        $perPage = $request->get('per_page', 20);
        $sales = $query->orderBy('id', 'desc')->paginate($perPage);

        // Prepare response data with decoded products
        $data = [];
        foreach ($sales->items() as $item) {
            $itemArr = $item->toArray();
            $itemArr['products'] = json_decode($itemArr['products'], true);
            $data[] = $itemArr;
        }

        return response()->json([
            'status' => true,
            'message' => 'Sales report retrieved successfully',
            'data' => $data,
            'pagination' => [
                'limit_page' => $perPage,
                'total_count' => $sales->total(),
                'total_page' => $sales->lastPage(),
                'current_page' => $sales->currentPage(),
                'current_page_count' => $sales->count(),
                'next_page' => $sales->hasMorePages() ? $sales->currentPage() + 1 : null,
                'previous_page' => $sales->onFirstPage() ? null : $sales->currentPage() - 1,
            ],
        ]);
    }
}
