<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\Sales;
use App\Models\Purchase;
use Carbon\Carbon;
use DB;

class ReportController extends Controller
{
    /**
     * Stock Report: shows current stock for each product
     * Filters:
     *  - search: product name or SKU
     */
    public function stock(Request $request)
    {
        $query = Product::query();

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('sku', 'like', "%{$search}%");
            });
        }

        // If 'days' input is provided, filter products created during the last 'days' days from today
        if ($request->filled('days')) {
            $days = (int) $request->get('days');
            if ($days > 0) {
                $fromDate = Carbon::now()->subDays($days)->startOfDay();
                $toDate = Carbon::now()->endOfDay();
                $query->whereBetween('created_at', [$fromDate, $toDate]);
            }
        }

        $perPage = $request->get('per_page', 20);
        $products = $query->orderBy('id', 'desc')->paginate($perPage);

        $data = [];
        foreach ($products->items() as $product) {
            $data[] = [
                'id' => $product->id,
                'name' => $product->name,
                'sku' => $product->sku,
                'stock' => $product->stock,
                'alert_quantity' => $product->alert_quantity,
                'price' => $product->price,
                'created_at' => $product->created_at,
            ];
        }

        return response()->json([
            'status' => true,
            'message' => 'Stock report retrieved successfully',
            'data' => $data,
            'pagination' => [
                'limit_page' => $perPage,
                'total_count' => $products->total(),
                'total_page' => $products->lastPage(),
                'current_page' => $products->currentPage(),
                'current_page_count' => $products->count(),
                'next_page' => $products->hasMorePages() ? $products->currentPage() + 1 : null,
                'previous_page' => $products->onFirstPage() ? null : $products->currentPage() - 1,
            ]
        ]);
    }


    /**
     * Profit-Loss Report: summarizes total purchase, sales, and computes profit/loss for given period.
     * Filters:
     *  - from_date: sales/purchase_date >= from_date
     *  - to_date:   sales/purchase_date <= to_date
     *  - search: product name/SKU (in products array details)
     */
    public function profitLoss(Request $request)
    {
        $from = $request->get('from_date');
        $to = $request->get('to_date');
        $search = $request->get('search');

        // Add support for 'days' input like in the stock() method
        if ($request->filled('days')) {
            $days = (int)$request->get('days');
            if ($days > 0) {
                $fromDate = Carbon::now()->subDays($days)->startOfDay()->toDateString();
                $toDate = Carbon::now()->endOfDay()->toDateString();

                // Only override if not custom from/to provided
                if (!$from) $from = $fromDate;
                if (!$to) $to = $toDate;
            }
        }

        // Gather all sales in the period and decode product lines
        $salesQuery = Sales::query();
        if ($from) $salesQuery->whereDate('sales_date', '>=', $from);
        if ($to) $salesQuery->whereDate('sales_date', '<=', $to);
        $salesLines = [];
        foreach ($salesQuery->get() as $sale) {
            $products = json_decode($sale->products, true);
            foreach ($products ?: [] as $prod) {
                $prod['sale_amount'] = isset($prod['subtotal']) ? $prod['subtotal'] : 0;
                $prod['sales_date'] = $sale->sales_date;
                $prod['sales_id'] = $sale->sales_id;
                $salesLines[] = $prod;
            }
        }

        // Gather all purchases in the period and decode product lines
        $purchaseQuery = Purchase::query();
        if ($from) $purchaseQuery->whereDate('purchase_date', '>=', $from);
        if ($to) $purchaseQuery->whereDate('purchase_date', '<=', $to);
        $purchaseLines = [];
        foreach ($purchaseQuery->get() as $purchase) {
            $products = json_decode($purchase->products, true);
            foreach ($products ?: [] as $prod) {
                $prod['purchase_amount'] = isset($prod['subtotal']) ? $prod['subtotal'] : 0;
                $prod['purchase_date'] = $purchase->purchase_date;
                $prod['purchase_id'] = $purchase->purchase_id;
                $purchaseLines[] = $prod;
            }
        }

        // Optional: search filter on product name/SKU
        if ($search) {
            // Get product IDs matching name/SKU
            $matchedProducts = Product::where('name', 'like', "%{$search}%")
                ->orWhere('sku', 'like', "%{$search}%")
                ->pluck('id')
                ->toArray();
            $salesLines = array_filter($salesLines, function($line) use ($matchedProducts) {
                return in_array($line['product_id'] ?? null, $matchedProducts);
            });
            $purchaseLines = array_filter($purchaseLines, function($line) use ($matchedProducts) {
                return in_array($line['product_id'] ?? null, $matchedProducts);
            });
        }

        // Summarize total sales, total purchases
        $totalSales = array_sum(array_column($salesLines, 'sale_amount'));
        $totalPurchases = array_sum(array_column($purchaseLines, 'purchase_amount'));
        $profit = $totalSales - $totalPurchases;

        $result = [
            'total_sales' => $totalSales,
            'total_purchases' => $totalPurchases,
            'profit' => $profit,
        ];

        return response()->json([
            'status' => true,
            'message' => 'Profit-Loss report retrieved successfully',
            'data' => $result,
        ]);
    }
}
