<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\Category;
use App\Models\Brand;
use App\Models\Unit;
use App\Models\Color;
use App\Models\Size;
use App\Models\Subcategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class ProductController extends Controller
{
    // 📝 List products with search, filter, sort, and pagination
    public function index(Request $request)
    {
        try {
            $query = Product::query();

            // 🔍 Search by name, barcode (partial - matches by any segment), or material
            if ($request->filled('search')) {
                $search = $request->search;
                $query->where(function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                      // Search barcode in segments separated by dot (.)
                      ->orWhereRaw("CONCAT('.',barcode,'.') LIKE ?", ["%.{$search}.%"])
                      ->orWhere('material', 'like', "%{$search}%");
                });
            }

            // 📁 Category-wise filter
            if ($request->filled('category_id')) {
                $query->where('cat_id', $request->category_id);
            }

            // 🧩 Status filter
            if ($request->filled('status') && in_array($request->status, ['active', 'inactive'])) {
                $query->where('status', $request->status);
            }

            // ↕️ Sorting
            $order = $request->get('order', 'desc');
            $query->orderBy('id', $order);

            // 📄 Pagination
            $perPage = $request->get('per_page', 10);
            $products = $query->paginate($perPage);

            // Transform products
            $products->getCollection()->transform(function ($product) {
                return [
                    'id' => $product->id,
                    'name' => $product->name,
                    'sku' => $product->sku,
                    'barcode' => $product->barcode,
                    'image' => $product->image ? asset('storage/' . $product->image) : null,
                    'category' => $product->cat_id ? Category::find($product->cat_id)->name ?? null : null,
                    'brand' => $product->brand_id ? Brand::find($product->brand_id)->name ?? null : null,
                    'unit' => $product->unit_id ? Unit::find($product->unit_id)->name ?? null : null,
                    'color' => $product->color_id ? Color::find($product->color_id)->name ?? null : null,
                    'price' => $product->price,
                    'offer_price' => $product->offer_price,
                    'stock' => $product->stock,
                    'alert_quantity' => $product->alert_quantity,
                    'weight' => $product->weight,
                    'sizes' => $product->size_ids ? Size::whereIn('id', explode(',', $product->size_ids))->pluck('name') : [],
                    'material' => $product->material,
                    'origin' => $product->origin,
                    'description' => $product->description,
                    'gallery_images' => $product->gallery_images
                        ? array_map(fn($img) => asset('storage/' . $img), json_decode($product->gallery_images))
                        : [],
                    'extra_status' => $product->extra_status,
                    'status' => $product->status,
                    'created_at' => $product->created_at->toDateTimeString(),
                    'updated_at' => $product->updated_at->toDateTimeString(),
                ];
            });

            return response()->json([
                'status' => true,
                'message' => 'Products retrieved successfully',
                'pagination' => [
                    'limit_page' => $perPage,
                    'total_count' => $products->total(),
                    'total_page' => $products->lastPage(),
                    'current_page' => $products->currentPage(),
                    'current_page_count' => $products->count(),
                    'next_page' => $products->hasMorePages() ? $products->currentPage() + 1 : null,
                    'previous_page' => $products->onFirstPage() ? null : $products->currentPage() - 1,
                ],
                'data' => $products->items(),
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    // 📝 Store new product
    public function store(Request $request)
    {
        try {
            $request->validate([
                'name' => 'required|string|max:255',
                'barcode' => 'nullable|string|max:255',
                'image' => 'required|image|mimes:jpeg,jpg,png,bmp,gif,svg,webp,heic,heif,tiff,tif,ico,avif|max:20480',
                'cat_id' => 'required|exists:categories,id',
                'subcat_id'=> 'nullable|exists:subcategories,id',
                'brand_id' => 'required|exists:brands,id',
                'unit_id' => 'required|exists:units,id',
                'color_id' => 'required|exists:colors,id',
                'price' => 'required|numeric|min:0',
                'offer_price' => 'required|numeric|min:0',
                'stock' => 'required|numeric|min:0',
                'alert_quantity' => 'required|numeric|min:0',
                'weight' => 'required|string|max:50',
                'size_ids' => 'required|string', // comma-separated size IDs
                'material' => 'required|string|max:255',
                'origin' => 'required|string|max:255',
                'description' => 'required|string',
                'gallery_images.*' => 'required|image|mimes:jpeg,jpg,png,bmp,gif,svg,webp,heic,heif,tiff,tif,ico,avif|max:20480',
                'extra_status' => 'nullable|string',
                'status' => 'required|in:active,inactive',
            ]);

            // Generate an 18-digit SKU with uppercase letters and numbers
            $sku = strtoupper(substr(bin2hex(random_bytes(18)), 0, 18));

            // Main image
            $imagePath = $request->hasFile('image') ? $request->file('image')->store('products', 'public') : null;

            // Gallery images
            $galleryPaths = [];
            if ($request->hasFile('gallery_images')) {
                foreach ($request->file('gallery_images') as $img) {
                    $galleryPaths[] = $img->store('products/gallery', 'public');
                }
            }

            $product = Product::create([
                'name' => $request->name,
                'sku' => $sku,
                'barcode' => $request->barcode,
                'image' => $imagePath,
                'cat_id' => $request->cat_id,
                'subcat_id' => $request->subcat_id,
                'brand_id' => $request->brand_id,
                'unit_id' => $request->unit_id,
                'color_id' => $request->color_id,
                'price' => $request->price,
                'offer_price' => $request->offer_price,
                'stock' => $request->stock,
                'alert_quantity' => $request->alert_quantity,
                'weight' => $request->weight,
                'size_ids' => $request->size_ids,
                'material' => $request->material,
                'origin' => $request->origin,
                'description' => $request->description,
                'gallery_images' => $galleryPaths ? json_encode($galleryPaths) : null,
                'extra_status' => $request->extra_status,
                'status' => $request->status ?? 'active',
            ]);

            // Barcode: concatenate all relevant IDs separated by dots so that search by any "part" is possible
            $barcodeParts = [
                '1336',
                $product->cat_id ?? '0',
                $product->subcat_id ?? '0',
                $product->brand_id ?? '0',
                $product->color_id ?? '0',
                $request->size_ids ?? '0',
                $product->id ?? '0',
            ];
            // To maximize searchability, ensure leading/trailing dots are present and all parts (incl. 0 values)
            $barcode = implode('.', $barcodeParts);

            // Update the product's barcode as segmentable
            $product->barcode = $barcode;
            $product->save();

            return response()->json(['status' => true, 'message' => 'Product created successfully', 'data' => $product]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    // 📝 Show single product
    public function show($sku)
    {
        try {
            $product = Product::where('sku',$sku)->first();
            if (!$product) return response()->json(['status'=>false,'message'=>'Product not found'], 404);

            $productData = [
                'id' => $product->id,
                'name' => $product->name,
                'sku'=> $product->sku,
                'barcode' => $product->barcode,
                'image' => $product->image ? asset('storage/' . $product->image) : null,
                'category' => $product->cat_id ? Category::find($product->cat_id)->name ?? null : null,
                'subcategory' => $product->cat_id ? Subcategory::find($product->cat_id)->name ?? null : null,
                'brand' => $product->brand_id ? Brand::find($product->brand_id)->name ?? null : null,
                'unit' => $product->unit_id ? Unit::find($product->unit_id)->name ?? null : null,
                'color' => $product->color_id ? Color::find($product->color_id)->name ?? null : null,
                'price' => $product->price,
                'offer_price' => $product->offer_price,
                'stock' => $product->stock,
                'alert_quantity' => $product->alert_quantity,
                'weight' => $product->weight,
                'sizes' => $product->size_ids ? Size::whereIn('id', explode(',', $product->size_ids))->pluck('name') : [],
                'material' => $product->material,
                'origin' => $product->origin,
                'description' => $product->description,
                'gallery_images' => $product->gallery_images
                    ? array_map(fn($img) => asset('storage/' . $img), json_decode($product->gallery_images))
                    : [],
                'extra_status' => $product->extra_status,
                'status' => $product->status,
                'created_at' => $product->created_at->toDateTimeString(),
                'updated_at' => $product->updated_at->toDateTimeString(),
            ];

            return response()->json(['status'=>true,'data'=>$productData]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    // 📝 Update product
    public function update(Request $request, $id)
    {
        try {
            $product = Product::find($id);
            if (!$product) return response()->json(['status'=>false,'message'=>'Product not found'], 404);

            $request->validate([
                'name' => 'nullable|string|max:255',
                'barcode' => 'nullable|numeric|unique:products,barcode,' . $id,
                'image' => 'nullable|image|mimes:jpeg,jpg,png,bmp,gif,svg,webp,heic,heif,tiff,tif,ico,avif|max:20480',
                'cat_id' => 'nullable|exists:categories,id',
                'subcat_id'=> 'nullable|exists:subcategories,id',
                'brand_id' => 'nullable|exists:brands,id',
                'unit_id' => 'nullable|exists:units,id',
                'color_id' => 'nullable|exists:colors,id',
                'price' => 'nullable|numeric|min:0',
                'offer_price' => 'nullable|numeric|min:0',
                'stock' => 'nullable|numeric|min:0',
                'alert_quantity' => 'nullable|numeric|min:0',
                'weight' => 'nullable|string|max:50',
                'size_ids' => 'nullable|string',
                'material' => 'nullable|string|max:255',
                'origin' => 'nullable|string|max:255',
                'description' => 'nullable|string',
                'gallery_images.*' => 'nullable|image|mimes:jpeg,jpg,png,bmp,gif,svg,webp,heic,heif,tiff,tif,ico,avif|max:20480',
                'extra_status' => 'nullable|string',
                'status' => 'nullable|in:active,inactive',
            ]);

            // Update main image
            if ($request->hasFile('image')) {
                if ($product->image) Storage::disk('public')->delete($product->image);
                $product->image = $request->file('image')->store('products', 'public');
            }

            // Add new images to gallery (append, don't replace)
            if ($request->hasFile('gallery_images')) {
                $existingGallery = $product->gallery_images ? json_decode($product->gallery_images, true) : [];
                foreach ($request->file('gallery_images') as $img) {
                    $existingGallery[] = $img->store('products/gallery', 'public');
                }
                $product->gallery_images = json_encode($existingGallery);
            }

            // Update other fields
            $fields = [
                'name', 'barcode', 'cat_id', 'subcat_id', 'brand_id', 'unit_id', 'color_id',
                'price', 'offer_price', 'stock', 'alert_quantity', 'weight',
                'size_ids', 'material', 'origin', 'description', 'extra_status', 'status'
            ];
            foreach ($fields as $field) {
                if ($request->has($field)) {
                    $product->$field = $field === 'extra_status' ? json_encode($request->$field) : $request->$field;
                }
            }

            $product->save();


            return response()->json(['status'=>true,'message'=>'Product updated successfully','data'=>$product]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    // 📝 Delete product
    public function destroy($id)
    {
        try {
            $product = Product::find($id);
            if (!$product) return response()->json(['status'=>false,'message'=>'Product not found'], 404);

            // Delete images
            if ($product->image) Storage::disk('public')->delete($product->image);
            if ($product->gallery_images) {
                foreach (json_decode($product->gallery_images) as $img) {
                    Storage::disk('public')->delete($img);
                }
            }

            $product->delete();
            return response()->json(['status'=>true,'message'=>'Product deleted successfully']);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }
}
