<?php

namespace App\Http\Controllers;

use App\Models\Expenses;
use Illuminate\Http\Request;

class ExpenseController extends Controller
{
    /**
     * Display a listing of expenses.
     */
    public function index(Request $request)
    {
        try {
            $query = Expenses::query();

            // Optional filtering by title or expense_id
            if ($request->filled('search')) {
                $search = $request->search;
                $query->where(function ($q) use ($search) {
                    $q->where('title', 'like', "%{$search}%")
                        ->orWhere('expense_id', 'like', "%{$search}%");
                });
            }

            // Add 'days' input to filter from today back N days
            if ($request->filled('days')) {
                $days = (int) $request->get('days');
                if ($days > 0) {
                    $fromDate = now()->subDays($days)->startOfDay();
                    $toDate = now()->endOfDay();
                    $query->whereBetween('date', [$fromDate, $toDate]);
                }
            }

            // Pagination
            $perPage = $request->get('per_page', 10);
            $expenses = $query->orderBy('id', 'desc')->paginate($perPage);

            return response()->json([
                'status' => true,
                'message' => 'Expenses retrieved successfully',
                'data' => $expenses->items(),
                'pagination' => [
                    'limit_page' => $perPage,
                    'total_count' => $expenses->total(),
                    'total_page' => $expenses->lastPage(),
                    'current_page' => $expenses->currentPage(),
                    'current_page_count' => $expenses->count(),
                    'next_page' => $expenses->hasMorePages() ? $expenses->currentPage() + 1 : null,
                    'previous_page' => $expenses->onFirstPage() ? null : $expenses->currentPage() - 1,
                ],
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Store a newly created expense in storage.
     */
    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'title' => 'required|string',
                'description' => 'nullable|string',
                'amount' => 'required|numeric',
                'date' => 'required|date',
                'paid_to' => 'nullable|string',
            ]);

            // Generate expense_id: "EX" + 13 chars of capital letters and numbers
            do {
                $expenseId = 'EXP' . strtoupper(substr(bin2hex(random_bytes(12)), 0, 12));
            } while (Expenses::where('expense_id', $expenseId)->exists());

            $validated['expense_id'] = $expenseId;

            $expense = Expenses::create($validated);

            return response()->json([
                'status' => true,
                'message' => 'Expense created successfully',
                'data' => $expense
            ], 201);
        } catch (\Illuminate\Validation\ValidationException $ve) {
            return response()->json([
                'status' => false,
                'message' => 'Validation failed',
                'errors' => $ve->errors(),
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Display the specified expense.
     */
    public function show($id)
    {
        try {
            $expense = Expenses::find($id);

            if (!$expense) {
                return response()->json([
                    'status' => false,
                    'message' => 'Expense not found'
                ], 404);
            }

            return response()->json([
                'status' => true,
                'message' => 'Expense retrieved successfully',
                'data' => $expense
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Update the specified expense in storage.
     */
    public function update(Request $request, $id)
    {
        try {
            $expense = Expenses::find($id);
            if (!$expense) {
                return response()->json([
                    'status' => false,
                    'message' => 'Expense not found'
                ], 404);
            }

            $validated = $request->validate([
                'expense_id' => 'sometimes|required|string|unique:expenses,expense_id,' . $expense->id,
                'title' => 'sometimes|required|string',
                'description' => 'nullable|string',
                'amount' => 'sometimes|required|numeric',
                'date' => 'sometimes|required|date',
                'paid_to' => 'nullable|string',
            ]);

            $expense->update($validated);

            return response()->json([
                'status' => true,
                'message' => 'Expense updated successfully',
                'data' => $expense
            ]);
        } catch (\Illuminate\Validation\ValidationException $ve) {
            return response()->json([
                'status' => false,
                'message' => 'Validation failed',
                'errors' => $ve->errors(),
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Remove the specified expense from storage.
     */
    public function destroy($id)
    {
        try {
            $expense = Expenses::find($id);

            if (!$expense) {
                return response()->json([
                    'status' => false,
                    'message' => 'Expense not found'
                ], 404);
            }

            $expense->delete();

            return response()->json([
                'status' => true,
                'message' => 'Expense deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }
}
