<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Sales;
use App\Models\Purchase;
use App\Models\Product;
use App\Models\Customer;
use App\Models\User;
use App\Models\Expenses;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function index()
    {
        // Total Orders (count of sales records)
        $totalOrders = Sales::count();

        // Total Customers - count from customers table
        $totalCustomers = Customer::count();

        // Total Products
        $totalProducts = Product::count();

        // Total Sales (sum of grand_total from sales table)
        $totalSales = Sales::sum(DB::raw('CAST(grand_total AS DECIMAL(15,2))'));

        // Total Revenue (same as total sales)
        $totalRevenue = $totalSales;

        // Total Purchase (sum of grand_total from purchases table)
        $totalPurchase = Purchase::sum('grand_total');

        // Total Expenses
        $totalExpenses = Expenses::sum('amount');

        // Total Profit (sales - purchase - expenses)
        $totalProfit = $totalSales - $totalPurchase - $totalExpenses;

        // Sales Overview Monthly (group sales by month using grand_total)
        $salesOverviewMonthly = Sales::selectRaw('YEAR(created_at) as year, MONTH(created_at) as month, SUM(CAST(grand_total AS DECIMAL(15,2))) as total')
            ->groupByRaw('YEAR(created_at), MONTH(created_at)')
            ->orderByRaw('year desc, month desc')
            ->take(12)
            ->get();

        // Top Selling Products - parse products JSON from sales and aggregate
        $allSales = Sales::select('products')->get();
        $productSales = [];

        foreach ($allSales as $sale) {
            $products = json_decode($sale->products, true);
            if (is_array($products)) {
                foreach ($products as $product) {
                    if (isset($product['product_id']) && isset($product['quantity'])) {
                        $productId = $product['product_id'];
                        $quantity = (float) $product['quantity'];

                        if (!isset($productSales[$productId])) {
                            $productSales[$productId] = [
                                'product_id' => $productId,
                                'total_sold' => 0,
                                'product' => null
                            ];
                        }
                        $productSales[$productId]['total_sold'] += $quantity;
                    }
                }
            }
        }

        // Sort by total_sold descending and get top 5
        usort($productSales, function($a, $b) {
            return $b['total_sold'] <=> $a['total_sold'];
        });
        $topProducts = array_slice($productSales, 0, 5);

        // Load product details for top products
        $productIds = array_column($topProducts, 'product_id');
        $productsData = Product::whereIn('id', $productIds)->get()->keyBy('id');

        foreach ($topProducts as &$topProduct) {
            if (isset($productsData[$topProduct['product_id']])) {
                $topProduct['product'] = $productsData[$topProduct['product_id']];
            }
        }
        $topProducts = collect($topProducts);

        // Recent Sales (5 most recent orders)
        $recentOrders = Sales::select('id', 'sales_id', 'customer_name', 'customer_phone', 'grand_total', 'paid_by', 'created_at')
            ->orderBy('created_at', 'desc')
            ->take(5)
            ->get();

        return response()->json([
            'status' => true,
            'message' => 'Dashboard data retrieved successfully',
            'data' => [
                'totalSales' => $totalSales,
                'totalProfit' => $totalProfit,
                'totalPurchase' => $totalPurchase,
                'totalExpenses' => $totalExpenses,
                'totalRevenue' => $totalRevenue,
                'totalOrders' => $totalOrders,
                'totalCustomers' => $totalCustomers,
                'totalProducts' => $totalProducts,
                'salesOverviewMonthly' => $salesOverviewMonthly,
                'topSellingProducts' => $topProducts,
                'recentOrders' => $recentOrders,
            ]
        ]);
    }
}
