<?php

namespace App\Http\Controllers;

use App\Models\Customer;
use Illuminate\Http\Request;

class CustomerController extends Controller
{
    // List all customers
    public function index(Request $request)
    {
        try {
            $query = Customer::query();

            if ($request->has('search') && !empty($request->search)) {
                $search = $request->search;
                $query->where(function($q) use ($search) {
                    $q->where('name', 'like', '%' . $search . '%')
                      ->orWhere('phone', 'like', '%' . $search . '%');
                });
            }

            // Add orderBy logic: default by 'id' descending, but allow custom ordering via query params
            $orderDir = $request->get('order', 'desc');
            $allowedOrderDirs = ['asc', 'desc'];
            if (!in_array(strtolower($orderDir), $allowedOrderDirs)) {
                $orderDir = 'desc';
            }
            $query->orderBy('id', $orderDir);

            // Pagination logic similar to ProductController.phpS
            $perPage = (int) $request->get('limit_page', 15);
            $customers = $query->paginate($perPage);

            return response()->json([
                'status' => true,
                'message' => 'Customers retrieved successfully',
                'pagination' => [
                    'limit_page' => $perPage,
                    'total_count' => $customers->total(),
                    'total_page' => $customers->lastPage(),
                    'current_page' => $customers->currentPage(),
                    'current_page_count' => $customers->count(),
                    'next_page' => $customers->hasMorePages() ? $customers->currentPage() + 1 : null,
                    'previous_page' => $customers->onFirstPage() ? null : $customers->currentPage() - 1,
                ],
                'data' => $customers->items(),
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    // Store a new customer
    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'phone' => 'required|string|max:255|unique:customers,phone',
                'address' => 'nullable|string|max:255',
                'purchase_items' => 'nullable|string',
                'purchase_amount' => 'nullable|string',
                'total_purchase_items' => 'nullable|string',
                'total_purchase_amount' => 'nullable|string',
                'earned_points' => 'nullable|string',
                'status'=> 'nullable|string',
            ]);

            $customer = Customer::create($validated);

            return response()->json([
                'status' => true,
                'message' => 'Customer created successfully',
                'data' => $customer
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    // Show a single customer
    public function show($id)
    {
        try {
            $customer = Customer::find($id);
            if (!$customer) {
                return response()->json(['status' => false, 'message' => 'Customer not found'], 404);
            }
            return response()->json(['status' => true, 'data' => $customer]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    // Update a customer
    public function update(Request $request, $id)
    {
        try {
            $customer = Customer::find($id);
            if (!$customer) {
                return response()->json(['status' => false, 'message' => 'Customer not found'], 404);
            }

            $validated = $request->validate([
                'name' => 'nullable|string|max:255',
                'phone' => 'nullable|string|max:255',
                'address' => 'nullable|string|max:255',
                'purchase_items' => 'nullable|string',
                'purchase_amount' => 'nullable|string',
                'total_purchase_items' => 'nullable|string',
                'total_purchase_amount' => 'nullable|string',
                'earned_points' => 'nullable|string',
                'status'=> 'nullable|string',
            ]);

            $customer->update($validated);

            return response()->json([
                'status' => true,
                'message' => 'Customer updated successfully',
                'data' => $customer
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    // Delete a customer
    public function destroy($id)
    {
        try {
            $customer = Customer::find($id);
            if (!$customer) {
                return response()->json(['status' => false, 'message' => 'Customer not found'], 404);
            }

            $customer->delete();

            return response()->json([
                'status' => true,
                'message' => 'Customer deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }
}
