<?php

namespace App\Http\Controllers;

use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class CategoryController extends Controller
{
    /**
     * Display a listing of categories.
     */
    public function index(Request $request)
    {
        try {
            $query = Category::query();

            // 🔍 Search filter
            if ($request->filled('search')) {
                $search = $request->search;
                $query->where('name', 'like', "%{$search}%");
            }

            // 🧩 Status filter
            if ($request->filled('status') && in_array($request->status, ['active', 'inactive'])) {
                $query->where('status', $request->status);
            }

            // ↕️ Sorting (default: desc)
            $order = $request->get('order', 'desc');
            $query->orderBy('id', $order);

            // 📄 Pagination (default: 10 per page)
            $perPage = $request->get('per_page', 10);
            $categories = $query->paginate($perPage);

            // 🖼️ Transform category data
            $categories->getCollection()->transform(function ($category) {
                return [
                    'id' => $category->id,
                    'name' => $category->name,
                    'status' => $category->status,
                    'image' => $category->image ? asset('storage/' . $category->image) : null,
                ];
            });

            return response()->json([
                'status' => true,
                'message' => 'Categories retrieved successfully',
                'pagination' => [
                    'limit_page' => $perPage,
                    'total_count' => $categories->total(),
                    'total_page' => $categories->lastPage(),
                    'current_page' => $categories->currentPage(),
                    'current_page_count' => $categories->count(),
                    'next_page' => $categories->hasMorePages() ? $categories->currentPage() + 1 : null,
                    'previous_page' => $categories->onFirstPage() ? null : $categories->currentPage() - 1,
                ],
                'data' => $categories->items(),
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Store a newly created category.
     */
    public function store(Request $request)
    {
        try {
            $request->validate([
                'name' => 'required|string|max:255',
                'status' => 'nullable|in:active,inactive',
                'image' => 'nullable|image|mimes:jpeg,jpg,png,bmp,gif,svg,webp,heic,heif,tiff,tif,ico,avif|max:20480',
            ]);

            $imagePath = null;
            if ($request->hasFile('image')) {
                $imagePath = $request->file('image')->store('categories', 'public');
            }

            $category = Category::create([
                'name' => $request->name,
                'status' => $request->status ?? 'active',
                'image' => $imagePath,
            ]);

            $categoryData = [
                'id' => $category->id,
                'name' => $category->name,
                'status' => $category->status,
                'image' => $category->image ? asset('storage/' . $category->image) : null,
            ];

            return response()->json(['status'=>true,'message' => 'Category created successfully', 'data' => $categoryData], 201);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Display the specified category.
     */
    public function show($id)
    {
        try {
            $category = Category::find($id);

            if (!$category) {
                return response()->json(['message' => 'Category not found'], 404);
            }

            $categoryData = [
                'id' => $category->id,
                'name' => $category->name,
                'status' => $category->status,
                'image' => $category->image ? asset('storage/' . $category->image) : null,
            ];

            return response()->json(['status'=>true,'data' => $categoryData]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Update the specified category.
     */
    public function update(Request $request, $id)
    {
        try {
            $category = Category::find($id);

            if (!$category) {
                return response()->json(['message' => 'Category not found'], 404);
            }

            $request->validate([
                'name' => 'nullable|string|max:255',
                'status' => 'nullable|in:active,inactive',
                'image' => 'nullable|image|mimes:jpeg,jpg,png,bmp,gif,svg,webp,heic,heif,tiff,tif,ico,avif|max:20480',
            ]);

            if ($request->hasFile('image')) {
                // Delete old image
                if ($category->image) {
                    Storage::disk('public')->delete($category->image);
                }

                $category->image = $request->file('image')->store('categories', 'public');
            }

            $category->name = $request->name ?? $category->name;
            $category->status = $request->status ?? $category->status;
            $category->save();

            $categoryData = [
                'id' => $category->id,
                'name' => $category->name,
                'status' => $category->status,
                'image' => $category->image ? asset('storage/' . $category->image) : null,
            ];

            return response()->json(['status'=>true,'message' => 'Category updated successfully', 'data' => $categoryData]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Remove the specified category.
     */
    public function destroy($id)
    {
        try {
            $category = Category::find($id);

            if (!$category) {
                return response()->json(['message' => 'Category not found'], 404);
            }

            // Delete image
            if ($category->image) {
                Storage::disk('public')->delete($category->image);
            }

            $category->delete();

            return response()->json(['status'=>true,'message' => 'Category deleted successfully']);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }
}
