<?php

namespace App\Http\Controllers;

use App\Models\Brand;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;

class BrandController extends Controller
{
    /**
     * Display all brands.
     */
    public function index(Request $request)
    {
        try {
            $query = Brand::query();

            // 🔍 Search filter
            if ($request->filled('search')) {
                $search = $request->search;
                $query->where(function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                        ->orWhere('description', 'like', "%{$search}%");
                });
            }

            // 🧩 Status filter
            if ($request->filled('status') && in_array($request->status, ['active', 'inactive'])) {
                $query->where('status', $request->status);
            }

            // ↕️ Sorting (default: desc)
            $order = $request->get('order', 'desc');
            $query->orderBy('id', $order);

            // 📄 Pagination (default: 10 per page)
            $perPage = $request->get('per_page', 10);
            $brands = $query->paginate($perPage);

            // 🖼️ Transform brand data
            $brands->getCollection()->transform(function ($brand) {
                return [
                    'id' => $brand->id,
                    'name' => $brand->name,
                    'description' => $brand->description,
                    'status' => $brand->status,
                    'logo' => $brand->logo ? asset('storage/' . $brand->logo) : null,
                    'created_at' => $brand->created_at,
                    'updated_at' => $brand->updated_at,
                ];
            });

            return response()->json([
                'status' => true,
                'message' => 'Brands retrieved successfully',
                'pagination' => [
                    'limit_page' => $perPage,
                    'total_count' => $brands->total(),
                    'total_page' => $brands->lastPage(),
                    'current_page' => $brands->currentPage(),
                    'current_page_count' => $brands->count(),
                    'next_page' => $brands->hasMorePages() ? $brands->currentPage() + 1 : null,
                    'previous_page' => $brands->onFirstPage() ? null : $brands->currentPage() - 1,
                ],
                'data' => $brands->items(),
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Store a new brand.
     */
    public function store(Request $request)
    {
        try {
            $request->validate([
                'name' => 'required|string|max:255|unique:brands,name',
                'logo' => 'nullable|image|mimes:jpeg,jpg,png,bmp,gif,svg,webp,heic,heif,tiff,tif,ico,avif|max:2048',
                'description' => 'nullable|string',
                'status' => 'in:active,inactive'
            ]);

            $data = [
                'name' => $request->name,
                'description' => $request->description,
                'status' => $request->status ?? 'active',
            ];

            if ($request->hasFile('logo')) {
                $path = $request->file('logo')->store('brands', 'public');
                $data['logo'] = $path;
            }

            $brand = Brand::create($data);

            return response()->json([
                'status' => true,
                'message' => 'Brand created successfully',
                'data' => $brand
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Display a single brand.
     */
    public function show($id)
    {
        try {
            $brand = Brand::find($id);

            if (!$brand) {
                return response()->json(['status' => false, 'message' => 'Brand not found'], 404);
            }

            $brand->logo = $brand->logo ? asset('storage/' . $brand->logo) : null;

            return response()->json(['status' => true, 'data' => $brand]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Update an existing brand.
     */
    public function update(Request $request, $id)
    {
        try {
            $brand = Brand::find($id);

            if (!$brand) {
                return response()->json(['status' => false, 'message' => 'Brand not found'], 404);
            }

            $request->validate([
                'name' => 'sometimes|string|max:255|unique:brands,name,' . $id,
                'logo' => 'nullable|image|mimes:jpeg,jpg,png,bmp,gif,svg,webp,heic,heif,tiff,tif,ico,avif|max:2048',
                'description' => 'nullable|string',
                'status' => 'in:active,inactive'
            ]);

            $brand->name = $request->name ?? $brand->name;
            $brand->description = $request->description ?? $brand->description;
            $brand->status = $request->status ?? $brand->status;

            if ($request->hasFile('logo')) {
                if ($brand->logo && Storage::disk('public')->exists($brand->logo)) {
                    Storage::disk('public')->delete($brand->logo);
                }
                $brand->logo = $request->file('logo')->store('brands', 'public');
            }

            $brand->save();

            $brand->logo = $brand->logo ? asset('storage/' . $brand->logo) : null;

            return response()->json([
                'status' => true,
                'message' => 'Brand updated successfully',
                'data' => $brand
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Delete a brand.
     */
    public function destroy($id)
    {
        try {
            $brand = Brand::find($id);

            if (!$brand) {
                return response()->json(['status' => false, 'message' => 'Brand not found'], 404);
            }

            if ($brand->logo && Storage::disk('public')->exists($brand->logo)) {
                Storage::disk('public')->delete($brand->logo);
            }

            $brand->delete();

            return response()->json(['status' => true, 'message' => 'Brand deleted successfully']);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }
}
