<?php

namespace App\Http\Controllers;

use App\Models\Banner;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class BannerController extends Controller
{
    // GET /banners
    public function index(Request $request)
    {
        try {
            $query = Banner::query();

            // 🔍 Search filter
            if ($request->filled('search')) {
                $search = $request->search;
                $query->where(function ($q) use ($search) {
                    $q->where('title', 'like', "%{$search}%")
                    ->orWhere('subtitle', 'like', "%{$search}%")
                    ->orWhere('description', 'like', "%{$search}%");
                });
            }

            // 🧩 Status filter
            if ($request->filled('status') && in_array($request->status, ['active', 'inactive'])) {
                $query->where('status', $request->status);
            }

            // ↕️ Sorting (default: desc)
            $order = $request->get('order', 'desc');
            $query->orderBy('id', $order);

            // 📄 Pagination (default: 10 per page)
            $perPage = $request->get('per_page', 10);
            $banners = $query->paginate($perPage);

            // 🖼️ Transform banner data
            $banners->getCollection()->transform(function ($banner) {
                return [
                    'id' => $banner->id,
                    'title' => $banner->title,
                    'subtitle' => $banner->subtitle,
                    'description' => $banner->description,
                    'btn_name' => $banner->btn_name,
                    'btn_link' => $banner->btn_link,
                    'status' => $banner->status,
                    'images' => $banner->image
                        ? array_map(fn($img) => asset('storage/' . $img), json_decode($banner->image))
                        : [],
                ];
            });

            return response()->json([
                'status' => true,
                'message' => 'Banners retrieved successfully',
                'pagination' => [
                    'limit_page' => $perPage,
                    'total_count' => $banners->total(),
                    'total_page' => $banners->lastPage(),
                    'current_page' => $banners->currentPage(),
                    'current_page_count' => $banners->count(),
                    'next_page' => $banners->hasMorePages() ? $banners->currentPage() + 1 : null,
                    'previous_page' => $banners->onFirstPage() ? null : $banners->currentPage() - 1,
                ],
                'data' => $banners->items(), // 👈 only paginated data (not metadata)
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    // POST /banners
    public function store(Request $request)
    {
        try {
            $request->validate([
                'images.*' => 'required|image|mimes:jpeg,jpg,png,bmp,gif,svg,webp,heic,heif,tiff,tif,ico,avif|max:20480',
                'title' => 'required|string|max:255',
                'subtitle' => 'required|string|max:255',
                'description' => 'required|string|max:255',
                'btn_name' => 'required|string|max:255',
                'btn_link' => 'required|url|max:255',
                'status' => 'required|in:active,inactive',
            ]);

            $banner = new Banner();

            if ($request->hasFile('images')) {
                $imagePaths = [];
                foreach ($request->file('images') as $image) {
                    $imagePaths[] = $image->store('banners', 'public');
                }
                $banner->image = json_encode($imagePaths);
            }

            $banner->title = $request->title;
            $banner->subtitle = $request->subtitle;
            $banner->description = $request->description;
            $banner->btn_name = $request->btn_name;
            $banner->btn_link = $request->btn_link;
            $banner->status = $request->status ?? 'active';
            $banner->save();

            return response()->json(['status'=>true, 'message' => 'Banner created successfully', 'banner' => $banner]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    // GET /banners/{id}
    public function show($id)
    {
        try {
            $banner = Banner::find($id);

            if (!$banner) {
                return response()->json(['message' => 'Banner not found'], 404);
            }

            $bannerData = [
                'id' => $banner->id,
                'title' => $banner->title,
                'subtitle' => $banner->subtitle,
                'description' => $banner->description,
                'btn_name' => $banner->btn_name,
                'btn_link' => $banner->btn_link,
                'status' => $banner->status,
                'images' => $banner->image
                    ? array_map(fn($img) => asset('storage/' . $img), json_decode($banner->image))
                    : [],
            ];

            return response()->json(['status'=>true, 'data' => $bannerData]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    // PUT/PATCH /banners/{id}
    public function update(Request $request, $id)
    {
        try {
            $banner = Banner::find($id);

            if (!$banner) {
                return response()->json(['message' => 'Banner not found'], 404);
            }

            $request->validate([
                'images.*' => 'nullable|mimes:jpeg,jpg,png,bmp,gif,svg,webp,heic,heif,tiff,tif,ico,avif|max:20480',
                'title' => 'nullable|string|max:255',
                'subtitle' => 'nullable|string|max:255',
                'description' => 'nullable|string|max:255',
                'btn_name' => 'nullable|string|max:255',
                'btn_link' => 'nullable|url|max:255',
                'status' => 'nullable|in:active,inactive',
            ]);

            if ($request->hasFile('images')) {
                // Delete old images
                if ($banner->image) {
                    foreach (json_decode($banner->image) as $oldImage) {
                        Storage::disk('public')->delete($oldImage);
                    }
                }

                $imagePaths = [];
                foreach ($request->file('images') as $image) {
                    $imagePaths[] = $image->store('banners', 'public');
                }
                $banner->image = json_encode($imagePaths);
            }

            $banner->title = $request->title ?? $banner->title;
            $banner->subtitle = $request->subtitle ?? $banner->subtitle;
            $banner->description = $request->description ?? $banner->description;
            $banner->btn_name = $request->btn_name ?? $banner->btn_name;
            $banner->btn_link = $request->btn_link ?? $banner->btn_link;
            $banner->status = $request->status ?? $banner->status;
            $banner->save();

            // Return banner with full image URLs
            $bannerResponse = [
                'id' => $banner->id,
                'title' => $banner->title,
                'subtitle' => $banner->subtitle,
                'description' => $banner->description,
                'btn_name' => $banner->btn_name,
                'btn_link' => $banner->btn_link,
                'status' => $banner->status,
                'images' => $banner->image ? array_map(fn($img) => asset('storage/' . $img), json_decode($banner->image)) : [],
            ];

            return response()->json(['status'=>true, 'message' => 'Banner updated successfully', 'banner' => $bannerResponse]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    // DELETE /banners/{id}
    public function destroy($id)
    {
        try {
            $banner = Banner::find($id);

            if (!$banner) {
                return response()->json(['message' => 'Banner not found'], 404);
            }

            if ($banner->image) {
                foreach (json_decode($banner->image) as $image) {
                    Storage::disk('public')->delete($image);
                }
            }

            $banner->delete();

            return response()->json(['status'=>true,'message' => 'Banner deleted successfully']);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }
}
